<?php
/**
 * Activity Ajax functions
 *
 * @since 3.0.0
 * @version 8.1.0
 */

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit;

add_action( 'admin_init', function() {
	$ajax_actions = array(
		array(
			'activity_filter' => array(
				'function' => 'bp_nouveau_ajax_object_template_loader',
				'nopriv'   => true,
			),
		),
		array(
			'get_single_activity_content' => array(
				'function' => 'bp_nouveau_ajax_get_single_activity_content',
				'nopriv'   => true,
			),
		),
		array(
			'activity_mark_pin' => array(
				'function' => 'bp_nouveau_ajax_mark_activity_pin',
				'nopriv'   => false,
			),
		),
		array(
			'activity_mark_unpin' => array(
				'function' => 'bp_nouveau_ajax_mark_activity_unpin',
				'nopriv'   => false,
			),
		),
		array(
			'activity_mark_fav' => array(
				'function' => 'bp_nouveau_ajax_mark_activity_favorite',
				'nopriv'   => false,
			),
		),
		array(
			'activity_mark_unfav' => array(
				'function' => 'bp_nouveau_ajax_unmark_activity_favorite',
				'nopriv'   => false,
			),
		),
		array(
			'activity_clear_new_mentions' => array(
				'function' => 'bp_nouveau_ajax_clear_new_mentions',
				'nopriv'   => false,
			),
		),
		array(
			'delete_activity' => array(
				'function' => 'bp_nouveau_ajax_delete_activity',
				'nopriv'   => false,
			),
		),
		array(
			'new_activity_comment' => array(
				'function' => 'bp_nouveau_ajax_new_activity_comment',
				'nopriv'   => false,
			),
		),
		array(
			'bp_nouveau_get_activity_objects' => array(
				'function' => 'bp_nouveau_ajax_get_activity_objects',
				'nopriv'   => false,
			),
		),
		array(
			'post_update' => array(
				'function' => 'bp_nouveau_ajax_post_update',
				'nopriv'   => false,
			),
		),
		array(
			'bp_spam_activity' => array(
				'function' => 'bp_nouveau_ajax_spam_activity',
				'nopriv'   => false,
			),
		),
		array(
			'communityally_activity_upload_file' => array(
				'function' => 'communityally_ajax_upload_file',
				'nopriv'   => false,
			),
		),
		array(
			'edit_activity_post' => array(
				'function' => 'bp_nouveau_ajax_post_update',
				'nopriv'   => false,
			),
		),
	);

	foreach ( $ajax_actions as $ajax_action ) {
		$action = key( $ajax_action );

		add_action( 'wp_ajax_' . $action, $ajax_action[ $action ]['function'] );

		if ( ! empty( $ajax_action[ $action ]['nopriv'] ) ) {
			add_action( 'wp_ajax_nopriv_' . $action, $ajax_action[ $action ]['function'] );
		}
	}
}, 12 );

/**
 * Pin an activity to the top via a POST request.
 *
 * @since CommunityAlly 1.0.5
 *
 * @return string JSON reply
 */
function bp_nouveau_ajax_mark_activity_pin() {
	if ( ! bp_is_post_request() ) {
		wp_send_json_error();
	}

	if ( empty($_POST['id']) || empty( $_POST['nonce'] ) || ! wp_verify_nonce( $_POST['nonce'], 'bp_nouveau_activity' ) ) {
		wp_send_json_error();
	}

	$activity_id   = intval($_POST['id']);
	$activity_item = new BP_Activity_Activity( $activity_id );
	if ( ! bp_activity_user_can_read( $activity_item, bp_loggedin_user_id() ) ) {
		wp_send_json_error();
	}

	$activity_item->type = 'pinned_activity';
	$update_result = $activity_item->save();

	if (true === $update_result) {
		bp_activity_reset_cache_incrementor();
		wp_send_json_success(array(
			'label' => __('Unpin', 'communityally')
			)
			);
	} else {
		wp_send_json_error();
	}
}

/**
 * Unpin an activity from the top via a POST request.
 *
 * @since CommunityAlly 1.0.5
 *
 * @return string JSON reply
 */
function bp_nouveau_ajax_mark_activity_unpin() {
	if ( ! bp_is_post_request() ) {
		wp_send_json_error();
	}

	if ( empty($_POST['id']) || empty( $_POST['nonce'] ) || ! wp_verify_nonce( $_POST['nonce'], 'bp_nouveau_activity' ) ) {
		wp_send_json_error();
	}

	$activity_id   = intval($_POST['id']);
	$activity_item = new BP_Activity_Activity( $activity_id );
	if ( ! bp_activity_user_can_read( $activity_item, bp_loggedin_user_id() ) ) {
		wp_send_json_error();
	}

	$activity_item->type = 'activity_update';
	$update_result = $activity_item->save();

	if (true === $update_result) {
		bp_activity_reset_cache_incrementor();
		wp_send_json_success(array(
			'label' => __('Pin', 'communityally')
			)
			);
	} else {
		wp_send_json_error();
	}
}

/**
 * Mark an activity as a favourite via a POST request.
 *
 * @since 3.0.0
 *
 * @return string JSON reply
 */
function bp_nouveau_ajax_mark_activity_favorite() {
	if ( ! bp_is_post_request() ) {
		wp_send_json_error();
	}

	// Nonce check!
	if ( empty( $_POST['nonce'] ) || ! wp_verify_nonce( $_POST['nonce'], 'bp_nouveau_activity' ) ) {
		wp_send_json_error();
	}

	$activity_id   = (int) $_POST['id'];
	$activity_item = new BP_Activity_Activity( $activity_id );
	if ( ! bp_activity_user_can_read( $activity_item, bp_loggedin_user_id() ) ) {
		wp_send_json_error();
	}

	if ( bp_activity_add_user_favorite( $_POST['id'] ) ) {
		$response = array( 'content' => __('Favorite', 'buddypress') );

		if ( ! bp_is_user() ) {
			$fav_count = (int) bp_get_total_favorite_count_for_user( bp_loggedin_user_id() );

			if ( 1 === $fav_count ) {
				$response['directory_tab'] = '<li id="activity-favorites" data-bp-scope="favorites" data-bp-object="activity">
					<a href="' . bp_loggedin_user_domain() . bp_nouveau_get_component_slug( 'activity' ) . '/favorites/">
						' . esc_html__( 'My Favorites', 'buddypress' ) . '
					</a>
				</li>';
			} else {
				$response['fav_count'] = $fav_count;
			}
		}

		wp_send_json_success( $response );
	} else {
		wp_send_json_error();
	}
}

/**
 * Un-favourite an activity via a POST request.
 *
 * @since 3.0.0
 *
 * @return string JSON reply
 */
function bp_nouveau_ajax_unmark_activity_favorite() {
	if ( ! bp_is_post_request() ) {
		wp_send_json_error();
	}

	// Nonce check!
	if ( empty( $_POST['nonce'] ) || ! wp_verify_nonce( $_POST['nonce'], 'bp_nouveau_activity' ) ) {
		wp_send_json_error();
	}

	if ( bp_activity_remove_user_favorite( $_POST['id'] ) ) {
		$response = array( 'content' => __('Favorite', 'buddypress') );

		$fav_count = (int) bp_get_total_favorite_count_for_user( bp_loggedin_user_id() );

		if ( 0 === $fav_count && ! bp_is_single_activity() ) {
			$response['no_favorite'] = '<li><div class="bp-feedback bp-messages info">
				' . __( 'Sorry, there was no activity found. Please try a different filter.', 'buddypress' ) . '
			</div></li>';
		} else {
			$response['fav_count'] = $fav_count;
		}

		wp_send_json_success( $response );
	} else {
		wp_send_json_error();
	}
}

/**
 * Clear mentions if the directory tab is clicked
 *
 * @since 3.0.0
 *
 * @return string JSON reply
 */
function bp_nouveau_ajax_clear_new_mentions() {
	if ( ! bp_is_post_request() ) {
		wp_send_json_error();
	}

	// Nonce check!
	if ( empty( $_POST['nonce'] ) || ! wp_verify_nonce( $_POST['nonce'], 'bp_nouveau_activity' ) ) {
		wp_send_json_error();
	}

	bp_activity_clear_new_mentions( bp_loggedin_user_id() );
	wp_send_json_success();
}

/**
 * Deletes an Activity item/Activity comment item received via a POST request.
 *
 * @since 3.0.0
 *
 * @return string JSON reply.
 */
function bp_nouveau_ajax_delete_activity() {
	$response = array(
		'feedback' => sprintf(
			'<div class="bp-feedback bp-messages error">%s</div>',
			esc_html__( 'There was a problem when deleting. Please try again.', 'buddypress' )
		),
	);

	// Bail if not a POST action.
	if ( ! bp_is_post_request() ) {
		wp_send_json_error( $response );
	}

	// Nonce check!
	if ( empty( $_POST['_wpnonce'] ) || ! wp_verify_nonce( $_POST['_wpnonce'], 'bp_activity_delete_link' ) ) {
		wp_send_json_error( $response );
	}

	if ( ! is_user_logged_in() ) {
		wp_send_json_error( $response );
	}

	if ( empty( $_POST['id'] ) || ! is_numeric( $_POST['id'] ) ) {
		wp_send_json_error( $response );
	}

	$activity = new BP_Activity_Activity( (int) $_POST['id'] );

	// Check access.
	if ( ! bp_activity_user_can_delete( $activity ) ) {
		wp_send_json_error( $response );
	}

	/** This action is documented in bp-activity/bp-activity-actions.php */
	do_action( 'bp_activity_before_action_delete_activity', $activity->id, $activity->user_id );

	// Deleting an activity comment.
	if ( ! empty( $_POST['is_comment'] ) ) {
		if ( ! bp_activity_delete_comment( $activity->item_id, $activity->id ) ) {
			wp_send_json_error( $response );
		}

	// Deleting an activity.
	} else {
		if ( ! bp_activity_delete( array( 'id' => $activity->id, 'user_id' => $activity->user_id ) ) ) {
			wp_send_json_error( $response );
		}
	}

	/** This action is documented in bp-activity/bp-activity-actions.php */
	do_action( 'bp_activity_action_delete_activity', $activity->id, $activity->user_id );

	// The activity has been deleted successfully
	$response = array( 'deleted' => $activity->id );

	// If on a single activity redirect to user's home.
	if ( ! empty( $_POST['is_single'] ) ) {
		$response['redirect'] = bp_core_get_user_domain( $activity->user_id );
		bp_core_add_message( __( 'Activity deleted successfully', 'buddypress' ) );
	}

	wp_send_json_success( $response );
}

/**
 * Fetches an activity's full, non-excerpted content via a POST request.
 * Used for the 'Read More' link on long activity items.
 *
 * @since 3.0.0
 *
 * @return string JSON reply
 */
function bp_nouveau_ajax_get_single_activity_content() {
	$response = array(
		'feedback' => sprintf(
			'<div class="bp-feedback bp-messages error">%s</div>',
			esc_html__( 'There was a problem displaying the content. Please try again.', 'buddypress' )
		),
	);

	// Bail if not a POST action.
	if ( ! bp_is_post_request() ) {
		wp_send_json_error( $response );
	}

	// Nonce check!
	if ( empty( $_POST['nonce'] ) || ! wp_verify_nonce( $_POST['nonce'], 'bp_nouveau_activity' ) ) {
		wp_send_json_error( $response );
	}

	$activity_array = bp_activity_get_specific(
		array(
			'activity_ids'     => $_POST['id'],
			'display_comments' => 'stream',
		)
	);

	if ( empty( $activity_array['activities'][0] ) ) {
		wp_send_json_error( $response );
	}

	$activity = $activity_array['activities'][0];

	/**
	 * Fires before the return of an activity's full, non-excerpted content via a POST request.
	 *
	 * @since 3.0.0
	 *
	 * @param string $activity Activity content. Passed by reference.
	 */
	do_action_ref_array( 'bp_nouveau_get_single_activity_content', array( &$activity ) );

	// Activity content retrieved through AJAX should run through normal filters, but not be truncated.
	remove_filter( 'bp_get_activity_content_body', 'bp_activity_truncate_entry', 5 );

	/** This filter is documented in bp-activity/bp-activity-template.php */
	$content = apply_filters_ref_array(
		'bp_get_activity_content_body',
		array(
			$activity->content,
			&$activity
		)
	);

	wp_send_json_success( array( 'contents' => $content ) );
}

/**
 * Sanitize an individual attachment entry
 *
 * @since CommunityAlly 1.0.2
 *
 * @return sanitized attachment entry
 */
function bp_nouveau_sanitize_individual_attachment_entry($entry) {
	if (!is_array($entry)) {
		return false;
	}
	if (!isset($entry['name']) || !isset($entry['path'])) {
		return false;
	}
	return $entry;
}

/**
 * Retrieve and sanitize image / file attachment array
 *
 * @since CommunityAlly 1.0.2
 *
 * @return attachment array
 */
function bp_nouveau_process_attachment_value() {
	$attachment_meta = array();
	if (!empty($_POST['attachment']) && is_array($_POST['attachment'])) {
		$image_attachment = array();
		foreach ($_POST['attachment'] as $entry) {
			$sanitized_entry = bp_nouveau_sanitize_individual_attachment_entry($entry);
			if (!empty($sanitized_entry)) {
				$image_attachment []= $sanitized_entry;
			}
		}
		$attachment_meta['image'] = $image_attachment;
	}
	if (!empty($_POST['file-attachment']) && is_array($_POST['file-attachment'])) {
		$file_attachment = array();
		foreach ($_POST['file-attachment'] as $entry) {
			$sanitized_entry = bp_nouveau_sanitize_individual_attachment_entry($entry);
			if (!empty($sanitized_entry)) {
				$file_attachment []= $sanitized_entry;
			}
		}
		$attachment_meta['file'] = $file_attachment;
	}
	return $attachment_meta;
}

/**
 * Posts new Activity comments received via a POST request.
 *
 * @since 3.0.0
 *
 * @global BP_Activity_Template $activities_template
 *
 * @return string JSON reply
 */
function bp_nouveau_ajax_new_activity_comment() {
	global $activities_template;
	$bp = buddypress();

	$response = array(
		'feedback' => sprintf(
			'<div class="bp-feedback bp-messages error">%s</div>',
			esc_html__( 'There was an error posting your reply. Please try again.', 'buddypress' )
		),
	);

	// Bail if not a POST action.
	if ( ! bp_is_post_request() ) {
		wp_send_json_error( $response );
	}

	// Nonce check!
	if ( !isset($_POST['is_update']) || empty( $_POST['_wpnonce_new_activity_comment'] ) ) {
		wp_send_json_error( $response );
	}
	$is_editing_existing = 'yes' === $_POST['is_update'];
	// the nonce verifier depends on whether we are editing an existing comment or adding a new one
	$nonce_verifier = 'new_activity_comment';
	if ($is_editing_existing) {
		$nonce_verifier = 'new_edit_comment';
	}
	if (!wp_verify_nonce( $_POST['_wpnonce_new_activity_comment'], $nonce_verifier )) {
		wp_send_json_error( $response );
	}

	if ( ! is_user_logged_in() ) {
		wp_send_json_error( $response );
	}

	// If the comment has no text but does have an attachment, add '[attachment-only]' to the content. This will
	// be removed when the comment is rendered.
	if ( empty( $_POST['content'] ) && ( !empty( $_POST['attachment'] ) || !empty( $_POST['file-attachment'] ) ) ) {
		$_POST['content'] = '[attachment-only]';
	}

	if ( empty( $_POST['content'] ) ) {
		wp_send_json_error( array( 'feedback' => sprintf(
			'<div class="bp-feedback bp-messages error">%s</div>',
			esc_html__( 'Please do not leave the comment area blank.', 'buddypress' )
		) ) );
	}

	if ( empty( $_POST['form_id'] ) || empty( $_POST['comment_id'] ) || ! is_numeric( $_POST['form_id'] ) || ! is_numeric( $_POST['comment_id'] ) ) {
		wp_send_json_error( $response );
	}

	$activity_id   = (int) $_POST['form_id'];
	$activity_item = new BP_Activity_Activity( $activity_id );
	if ( ! bp_activity_user_can_read( $activity_item ) ) {
		wp_send_json_error( $response );
	}

	$comment_param = array(
		'activity_id' => $_POST['form_id'],
		'content' => CommunityAlly_Members_Setup::sanitize_mentioned_content(),
		'mentions' => CommunityAlly_Members_Setup::extract_mention_mapping()
	);
	$comment_id_input = $_POST['comment_id'];
	if ($is_editing_existing) {
		$comment_param['id'] = $comment_id_input;
		$activity = new BP_Activity_Activity($comment_id_input);
		$comment_param['parent_id'] = $activity->secondary_item_id;
		$comment_param['recorded_time'] = $activity->date_recorded;
		$comment_param['user_id'] = $activity->user_id; // need to keep the original poster
	} else {
		$comment_param['parent_id'] = $comment_id_input;
	}
	$comment_id = bp_activity_new_comment($comment_param);

	if ( ! $comment_id ) {
		if ( ! empty( $bp->activity->errors['new_comment'] ) && is_wp_error( $bp->activity->errors['new_comment'] ) ) {
			$response = array( 'feedback' => sprintf(
				'<div class="bp-feedback bp-messages error">%s</div>',
				esc_html( $bp->activity->errors['new_comment']->get_error_message() )
			) );
			unset( $bp->activity->errors['new_comment'] );
		}

		wp_send_json_error( $response );
	}

	// save attachment to activity meta
	$attachment_meta = bp_nouveau_process_attachment_value();
	if (!empty($attachment_meta) || $is_editing_existing) {
		bp_activity_update_meta($comment_id, 'communityally_attachment', $attachment_meta);
	}

	// Load the new activity item into the $activities_template global.
	bp_has_activities(
		array(
			'display_comments' => 'stream',
			'hide_spam'        => false,
			'show_hidden'      => true,
			'include'          => $comment_id,
		)
	);

	// Swap the current comment with the activity item we just loaded.
	if ( isset( $activities_template->activities[0] ) ) {
		$activities_template->activity                  = new stdClass();
		$activities_template->activity->id              = $activities_template->activities[0]->item_id;
		$activities_template->activity->current_comment = $activities_template->activities[0];

		// Because the whole tree has not been loaded, we manually
		// determine depth.
		$depth     = 1;
		$parent_id = (int) $activities_template->activities[0]->secondary_item_id;
		while ( $parent_id !== (int) $activities_template->activities[0]->item_id ) {
			$depth++;
			$p_obj     = new BP_Activity_Activity( $parent_id );
			$parent_id = (int) $p_obj->secondary_item_id;
		}
		$activities_template->activity->current_comment->depth = $depth;
	}

	ob_start();
	// Get activity comment template part.
	bp_get_template_part( 'activity/comment' );
	$response = array( 'contents' => ob_get_contents() );
	ob_end_clean();

	unset( $activities_template );

	wp_send_json_success( $response );
}

/**
 * Get items to attach the activity to.
 *
 * This is used within the activity post form autocomplete field.
 *
 * @since 3.0.0
 *
 * @return string JSON reply
 */
function bp_nouveau_ajax_get_activity_objects() {
	$response = array();

	if ( empty( $_POST['nonce'] ) || ! wp_verify_nonce( $_POST['nonce'], 'bp_nouveau_activity' ) ) {
		wp_send_json_error( $response );
	}

	if ( 'group' === $_POST['type'] ) {
		$groups = groups_get_groups(
			array(
				'user_id'      => bp_loggedin_user_id(),
				'search_terms' => $_POST['search'],
				'show_hidden'  => true,
				'per_page'     => 2,
			)
		);

		wp_send_json_success( array_map( 'bp_nouveau_prepare_group_for_js', $groups['groups'] ) );
	} else {

		/**
		 * Filters the response for custom activity objects.
		 *
		 * @since 3.0.0
		 *
		 * @param array $response Array of custom response objects to send to AJAX return.
		 * @param array $value    Activity object type from $_POST global.
		 */
		$response = apply_filters( 'bp_nouveau_get_activity_custom_objects', $response, $_POST['type'] );
	}

	if ( empty( $response ) ) {
		wp_send_json_error( array( 'error' => __( 'No activities were found.', 'buddypress' ) ) );
	} else {
		wp_send_json_success( $response );
	}
}

/**
 * Processes Activity updates received via a POST request.
 *
 * @since 3.0.0
 *
 * @return string JSON reply
 */
function bp_nouveau_ajax_post_update() {
	$bp = buddypress();

	if ( ! is_user_logged_in() || empty( $_POST['_wpnonce_post_update'] ) ) {
		wp_send_json_error();
	}
	$is_updating_existing = false;
	$nonce_verifier = 'post_update';
	if (!empty($_POST['is_update']) && 'yes' === $_POST['is_update']) {
		$is_updating_existing = true;
		$nonce_verifier = 'edit_existing_activity';
	}

	if (!wp_verify_nonce( $_POST['_wpnonce_post_update'], $nonce_verifier)) {
		wp_send_json_error();
	}

	// If the post has no text but does have an attachment, add '[attachment-only]' to the post content. This will
	// be removed when the post is rendered.
	if ( empty( $_POST['content'] ) && ( !empty($_POST['attachment']) || !empty($_POST['file-attachment']) ) ) {
		$_POST['content'] = '[attachment-only]';
	}

	if ( empty( $_POST['content'] ) ) {
		wp_send_json_error(
			array(
				'message' => __( 'Please enter some content to post.', 'buddypress' ),
			)
		);
	}

	// communityally_group shortcode in comments or updates should not get rendered
	$_POST['content'] = preg_replace('#\[communityally_group(.*)\]#', '&#91;communityally_group$1&#93;',$_POST['content']);

	$activity_id = 0;
	$item_id     = 0;
	$object      = '';
	$is_private  = false;

	// Try to get the item id from posted variables.
	if ( ! empty( $_POST['item_id'] ) ) {
		$item_id = (int) $_POST['item_id'];
	}

	// Try to get the object from posted variables.
	if ( ! empty( $_POST['object'] ) ) {
		$object = sanitize_key( $_POST['object'] );

	// If the object is not set and we're in a group, set the item id and the object
	} elseif ( bp_is_group() ) {
		$item_id = bp_get_current_group_id();
		$object  = 'group';
		$status  = groups_get_current_group()->status;
	}

	$activity_param = array(
		'content'  => CommunityAlly_Members_Setup::sanitize_mentioned_content(),
		'mentions' => CommunityAlly_Members_Setup::extract_mention_mapping()
	);
	if ( 'user' === $object && bp_is_active( 'activity' ) ) {
		$activity_id = bp_activity_post_update($activity_param);
	} elseif ($is_updating_existing) {
		$activity_id = $_POST['activity_id'];

		$activity = new BP_Activity_Activity($activity_id);
		if (empty($activity) || !communityally_activity_user_can_edit($activity)) {
			wp_send_json_error(
				array(
					'message' => __( 'There was a problem posting your update. Please try again.', 'buddypress' ),
				)
			);
		}
		$activity->content = $activity_param['content'];
		$activity->mentions = $activity_param['mentions'];
		$result = $activity->save();
		if (true !== $result) {
			wp_send_json_error(
				array(
					'message' => __( 'There was a problem posting your update. Please try again.', 'buddypress' ),
				)
			);
		}

		// save attachment to activity meta
		$attachment_meta = bp_nouveau_process_attachment_value();
		bp_activity_update_meta($activity_id, 'communityally_attachment', $attachment_meta);

		$is_private = $activity->hide_sitewide > 0;
	} elseif ( 'group' === $object ) {
		if ( $item_id && bp_is_active( 'groups' ) ) {
			// This function is setting the current group!
			$activity_param['group_id'] = $item_id;
			if (!communityally_user_can_access_group(bp_loggedin_user_id(), $item_id)) {
				wp_send_json_error(
					array(
						'message' => __( 'There was a problem posting your update. Please try again.', 'buddypress' ),
					)
				);
			}
			$activity_id = groups_post_update($activity_param);

			// save attachment to activity meta
			$attachment_meta = bp_nouveau_process_attachment_value();
			if (!empty($attachment_meta)) {
				bp_activity_update_meta($activity_id, 'communityally_attachment', $attachment_meta);
			}

			if ( empty( $status ) ) {
				if ( ! empty( $bp->groups->current_group->status ) ) {
					$status = $bp->groups->current_group->status;
				} else {
					$group  = groups_get_group( array( 'group_id' => $item_id ) );
					$status = $group->status;
				}

				$is_private = 'public' !== $status;
			}
		}
	} else {
		/** This filter is documented in bp-activity/actions/post.php */
		$activity_id = apply_filters( 'bp_activity_custom_update', false, $object, $item_id, $_POST['content'] );
	}

	if ( empty( $activity_id ) ) {
		wp_send_json_error(
			array(
				'message' => __( 'There was a problem posting your update. Please try again.', 'buddypress' ),
			)
		);
	}

	ob_start();
	if ( bp_has_activities( array( 'include' => $activity_id, 'show_hidden' => $is_private ) ) ) {
		while ( bp_activities() ) {
			bp_the_activity();
			bp_get_template_part('activity/entry', null, array('just_posted' => true));
		}
	}
	$acivity = ob_get_contents();
	ob_end_clean();

	wp_send_json_success( array(
		'id'           => $activity_id,
		'message'      => esc_html__( 'Update posted.', 'buddypress' ) . ' ' . sprintf( '<a href="%s" class="just-posted">%s</a>', esc_url( bp_activity_get_permalink( $activity_id ) ), esc_html__( 'View activity.', 'buddypress' ) ),
		'activity'     => $acivity,

		/**
		 * Filters whether or not an AJAX post update is private.
		 *
		 * @since 3.0.0
		 *
		 * @param string/bool $is_private Privacy status for the update.
		 */
		'is_private'   => apply_filters( 'bp_nouveau_ajax_post_update_is_private', $is_private ),
		'is_directory' => bp_is_activity_directory(),
	) );
}

/**
 * AJAX spam an activity item or comment.
 *
 * @since 3.0.0
 *
 * @return string JSON reply.
 */
function bp_nouveau_ajax_spam_activity() {
	$bp = buddypress();

	$response = array(
		'feedback' => sprintf(
			'<div class="bp-feedback bp-messages error">%s</div>',
			esc_html__( 'There was a problem marking this activity as spam. Please try again.', 'buddypress' )
		),
	);

	// Bail if not a POST action.
	if ( ! bp_is_post_request() ) {
		wp_send_json_error( $response );
	}

	if ( ! is_user_logged_in() || ! bp_is_active( 'activity' ) || empty( $bp->activity->akismet ) ) {
		wp_send_json_error( $response );
	}

	if ( empty( $_POST['id'] ) || ! is_numeric( $_POST['id'] ) ) {
		wp_send_json_error( $response );
	}

	// Is the current user allowed to spam items?
	if ( ! bp_activity_user_can_mark_spam() ) {
		wp_send_json_error( $response );
	}

	$activity = new BP_Activity_Activity( (int) $_POST['id'] );

	if ( empty( $activity->component ) ) {
		wp_send_json_error( $response );
	}

	// Nonce check!
	if ( empty( $_POST['_wpnonce'] ) || ! wp_verify_nonce( $_POST['_wpnonce'], 'bp_activity_akismet_spam_' . $activity->id ) ) {
		wp_send_json_error( $response );
	}

	/** This action is documented in bp-activity/bp-activity-actions.php */
	do_action( 'bp_activity_before_action_spam_activity', $activity->id, $activity );

	// Mark as spam.
	bp_activity_mark_as_spam( $activity );
	$activity->save();

	/** This action is documented in bp-activity/bp-activity-actions.php */
	do_action( 'bp_activity_action_spam_activity', $activity->id, $activity->user_id );

	// Prepare the successfull reply
	$response = array( 'spammed' => $activity->id );

	// If on a single activity redirect to user's home.
	if ( ! empty( $_POST['is_single'] ) ) {
		$response['redirect'] = bp_core_get_user_domain( $activity->user_id );
		bp_core_add_message( __( 'This activity has been marked as spam and is no longer visible.', 'buddypress' ) );
	}

	// Send the json reply
	wp_send_json_success( $response );
}

/**
 * Processes Activity updates received via a POST request.
 *
 * @since CommunityAlly 1.0.2
 *
 * @return string JSON reply
 */
function communityally_ajax_upload_file() {
	if ( ! bp_is_post_request() ) {
		wp_die();
	}

	check_admin_referer( 'bp-uploader' );

	// Sending the json response will be different if the current Plupload runtime is html4.
	$is_html4 = ! empty( $_POST['html4' ] );

	$file_attachment = new BP_Attachment_User_Upload();
	$uploaded = $file_attachment->upload( $_FILES );

	if ( ! empty( $uploaded['error'] ) ) {
		// Upload error response.
		bp_attachments_json_response( false, $is_html4, array(
			'type'    => 'upload_error',
			'message' => sprintf(
				/* translators: %s: the upload error message */
				__( 'Upload Failed! Error was: %s', 'buddypress' ),
				$uploaded['error']
			),
		) );
	}

	$error_message = __( 'There was a problem uploading the file.', 'communityally' );

	$bp_attachments_uploads_dir = bp_attachments_user_file_upload_dir();

	// The BP Attachments Uploads Dir is not set, stop.
	if ( ! $bp_attachments_uploads_dir || false === strpos($uploaded['file'], $bp_attachments_uploads_dir['path']) ) {
		bp_attachments_json_response( false, $is_html4, array(
			'type'    => 'upload_error',
			'message' => $error_message,
		) );
	}

	// only store the attachment file name, so the full path is NOT returned to client-side
	$attachment_file_name = str_replace($bp_attachments_uploads_dir['path'], '', $uploaded['file']);
	$attachment_file_name = ltrim($attachment_file_name, '/\\');	// trim slashes off the front of the name

	// Set the name of the file.
	$name       = $_FILES['file']['name'];

	// Finally return the attachment url to the UI.
	bp_attachments_json_response( true, $is_html4, array(
		'name'          => $name,
		'path'			=> $attachment_file_name,
		'url'           => $uploaded['url'],
		'feedback_code' => 1,
	) );
}