<?php

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit;

class CommunityAllySettings {
	const GENERAL_SETTINGS_KEY = '_communityally_general_settings';
	public static function show_settings() {
		if (!current_user_can('manage_options')) {
			wp_die('You do not have sufficient permissions to access this page.');
		}
		include (dirname(__FILE__) . '/settings-main-display.php');
	}

	// <editor-fold defaultstate="collapsed" desc="Add URL rewrite rules">
	public static function add_rewrite_tags() {
		add_rewrite_tag('%ca_subpage%', '([^/]*)', 'ca_subpage=');
	}
	public static function add_rewrite_rules() {
		CommunityAlly_Members_Setup::add_rewrite_rules();
		CommunityAlly_Groups_Setup::add_rewrite_rules();
	}
	public static function add_query_vars($vars) {
		$vars []= 'ca_subpage';
		$vars []= 'ca_user';
		return $vars;
	}
	// </editor-fold>

	// <editor-fold defaultstate="collapsed" desc="enqueue admin resources">
	public static function enqueue_administrative_resources($hook) {
		// do not include the http or https protocol in the ajax url
		$ajax_url = preg_replace("/^http:/i", '', admin_url('admin-ajax.php'));
		$ajax_url = preg_replace("/^https:/i", '', $ajax_url);

		$root_url = buddypress()->plugin_url;

		wp_enqueue_script('communityally-admin-notice', $root_url . 'settings/js/admin-notice.min.js', array('jquery'), CommunityAllyUpdater::VERSION);

		wp_localize_script('communityally-admin-notice', 'communityally_admin_notice_data_object',
			array('ajax_url' => $ajax_url));

		if (strpos($hook, self::GENERAL_SETTINGS_KEY) === false) {
			return;
		}

		wp_enqueue_script('communityally-settings-backend', $root_url . 'settings/settings.js', array(), CommunityAllyUpdater::VERSION);

		$script_data = array(
			'ajax_url' => $ajax_url,
			'user_profile_edit_url' => add_query_arg('tab', 'profile', admin_url('admin.php?page=_accessally_dir_base')),
			'nonce' => wp_create_nonce('communityally-update-nonce')
			);
		wp_localize_script('communityally-settings-backend', 'communityally_settings', $script_data);
		self::enqueue_administrative_chameleon_resources();

		wp_enqueue_style('communityally-settings-backend', $root_url . 'settings/settings.css',
			array('accessally-shared'), CommunityAllyUpdater::VERSION);
		wp_enqueue_editor();
	}

	public static function enqueue_administrative_chameleon_resources() {
		$root_url = buddypress()->plugin_url;
		wp_enqueue_script('communityally-chameleon', $root_url . 'settings/communityally-chameleon.js', false, CommunityAllyUpdater::VERSION);

		$license_settings = AccessAllySettingLicense::get_license_settings();
		$license_email = $license_settings['email'];
		$user_email = '';
		$current_user_id = get_current_user_id();
		if ($current_user_id > 0) {
			$current_user_data = get_userdata($current_user_id);
			if (!empty($current_user_data)) {
				$user_email = $current_user_data->user_email;
			}
		}
		wp_localize_script('communityally-chameleon', 'communityally_chameleon_data',
			array(
				'uid' => get_bloginfo('wpurl') . ' - ' . $user_email,
				'email' => $user_email,
				'license_email' => $license_email,
				'version' => CommunityAllyUpdater::VERSION
			));
	}
	// </editor-fold>

	// <editor-fold defaultstate="collapsed" desc="Add menu page">
	public static function add_accessally_submenu_pages($submenus) {
		$submenus['195-communityally'] = array(
			'title' => 'CommunityAlly General Settings',
			'name' => 'CommunityAlly',
			'slug' => self::GENERAL_SETTINGS_KEY,
			'icon-class' => 'dashicons-groups',
			'show_function' => array(__CLASS__, 'show_settings')
			);
		return $submenus;
	}
	// </editor-fold>

	// <editor-fold defaultstate="collapsed" desc="Add actions">
	public static function add_actions() {
		add_action('wp_ajax_communityally_special_pages_edit', array(__CLASS__, 'ajax_special_pages_edit_redirect'));
	}
	// </editor-fold>

	// <editor-fold defaultstate="collapsed" desc="General settings">
	private static $default_general_settings = array(
		'select-user-profile' => 'default',
		'select-members-page' => '',
		'members-page-slug' => '',
		'checked-enable-messages-component' => 'no',
		);
	private static $cached_general_settings = false;
	public static function get_general_settings() {
		if (false === self::$cached_general_settings) {
			$settings = get_option(self::GENERAL_SETTINGS_KEY, false);
			if (!is_array($settings)) {
				$settings = self::$default_general_settings;
			}

			self::$cached_general_settings = self::sanitize_general_settings($settings);
		}
		return self::$cached_general_settings;
	}
	private static function sanitize_general_settings($settings) {
		$settings = wp_parse_args($settings, self::$default_general_settings);
		return $settings;
	}
	private static function update_general_settings($settings) {
		$settings = self::sanitize_general_settings($settings);
		$settings = self::validate_special_page_settings($settings);

		// force flush rewrite rules so they can be regenerated
		delete_option('rewrite_rules');

		update_option(self::GENERAL_SETTINGS_KEY, $settings, 'no');
		self::$cached_general_settings = false;
		CommunityAlly_Members_Setup::clear_cache();

		if ('yes' === $settings['checked-enable-messages-component']) {
			CommunityAllyMessages::activate_messages_component();
		} else {
			CommunityAllyMessages::deactivate_messages_component();
		}
		return $settings;
	}
	// </editor-fold>

	// <editor-fold defaultstate="collapsed" desc="Validate special page settings">
	public static function validate_special_page_settings($settings = false) {
		$save_on_update = false;
		if (false === $settings) {
			$save_on_update = true;
			$settings = self::get_general_settings();
		}
		$need_update = false;
		$members_page = get_post($settings['select-members-page']);
		if (empty($members_page)) {
			$members_page_id = CommunityAlly_Members_Setup::create_new_members_page();

			$members_page = get_post($members_page_id);
			if (!empty($members_page)) {
				$settings['select-members-page'] = $members_page_id;
				$settings['members-page-slug'] = $members_page->post_name;
				$need_update = true;
			} else {
				$settings['members-page-slug'] = '';
			}
		} else {
			$settings['members-page-slug'] = $members_page->post_name;
		}

		if ($need_update) {
			if ($save_on_update) {
				self::update_general_settings($settings);
			}
		}
		return $settings;
	}
	// </editor-fold>

	// <editor-fold defaultstate="collapsed" desc="Get all page for special page selection">
	public static function get_page_selection_code() {
		global $wpdb;
		$posts = $wpdb->get_results("SELECT ID, post_title FROM $wpdb->posts WHERE post_status IN ('publish','draft') AND post_type = 'page' AND post_parent = 0 ORDER BY post_title", OBJECT_K);

		$code = '';
		foreach ($posts as $entry) {
			$code .= '<option s--{{name}}--' . $entry->ID . '--d value="' . $entry->ID . '">' . esc_html($entry->post_title) . ' (' . $entry->ID . ')</option>';
		}
		return $code;
	}
	// </editor-fold>

	// <editor-fold defaultstate="collapsed" desc="Show general settings">
	public static function show_general_settings() {
		$general_settings = self::get_general_settings();

		$code = file_get_contents(dirname(__FILE__) . '/settings-general.php');

		$accessally_user_profile_settings = apply_filters('accessally_get_user_profile_settings', false);
		$user_profile_selection_code = '';
		if (!empty($accessally_user_profile_settings)) {
			foreach ($accessally_user_profile_settings['profiles'] as $profile_id => $profile_settings) {
				$user_profile_selection_code .= '<option s--select-user-profile--' . $profile_id . '--d value="'.
					esc_attr($profile_id) . '">' . esc_html($profile_id . '. ' . $profile_settings['name']) . '</option>';
			}
		}
		$code = str_replace('{{accessally-profile-selection}}', $user_profile_selection_code, $code);

		$page_selection_code = self::get_page_selection_code();

		$members_page_selection_code = str_replace('{{name}}', 'select-members-page', $page_selection_code);
		$code = str_replace('{{members-page-selection}}', $members_page_selection_code, $code);

		$edit_url = add_query_arg('action', 'communityally_special_pages_edit', admin_url('admin-ajax.php', 'relative'));
		$code = str_replace('{{edit-url}}', esc_attr($edit_url), $code);

		// need to replace the value for the edit URL, as the select variable will not be replaced with the actual value in recursive_replace_real_values()
		$code = str_replace('{{select-members-page}}', $general_settings['select-members-page'], $code);

		$code = CommunityAllyUtilities::recursive_replace_real_values($code, $general_settings, '', false);
		$code = CommunityAllyUtilities::replace_all_toggle($code, $general_settings);

		return $code;
	}
	// </editor-fold>

	// <editor-fold defaultstate="collapsed" desc="Ajax callback: save page settings">
	public static function ajax_save_page_settings() {
		try{
			if (!isset($_POST['input']) || !isset($_POST['page']) ||
				!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'communityally-update-nonce')) {
				throw new Exception('The setting page is outdated. Please reload this page.');
			}
			$data = stripslashes($_POST['input']);
			$input = CommunityAllyUtilities::convert_setting_string_to_array($data);

			$result = array('status' => 'success', 'message' => 'Settings have been saved successfully');
			$page_slug = $_POST['page'];
			$code = array();
			if ('general' === $page_slug) {
				if (!empty($_POST['create_profile']) && 'yes' === $_POST['create_profile']) {
					$new_profile_id = apply_filters('accessally_create_user_profile', 'default');
					$input['select-user-profile'] = $new_profile_id;

					$result['message'] = 'A new User Profile has been created. Click on the Edit button to customize it.';
					$result['profile-id'] = $new_profile_id;
				}
				self::update_general_settings($input);

				$code['general'] = self::show_general_settings();
			}
			$result['code'] = $code;
			$result['warnings'] = '';
			echo json_encode($result);
		} catch (Exception $ex) {
			$error = array('status' => 'error', 'message' => 'Save settings failed due to error [' . $ex->getMessage() . '].');
			echo json_encode($error);
		}
		die();
	}
	// </editor-fold>

	// <editor-fold defaultstate="collapsed" desc="Add CommunityAlly shortcode to Shortcode Adder">
	public static function enqueue_shortcode_render_script($admin_url) {
		$root_url = buddypress()->plugin_url;
		wp_enqueue_script('communityally-shortcode-render', $root_url . 'settings/communityally-shortcode-render.js',
			array('jquery', 'jquery-ui-autocomplete'), CommunityAllyUpdater::VERSION);
		wp_localize_script('communityally-shortcode-render', 'communityally_shortcode_render',
			array('ajax_url' => $admin_url,
				'nonce' => wp_create_nonce('communityally-update-nonce')
			));

		wp_enqueue_style('communityally-shortcode-render', $root_url . 'settings/communityally-shortcode-render.css',
			false, CommunityAllyUpdater::VERSION);
	}

	private static function get_group_selection_code() {
		$template = '<option value="{{group-id}}">{{group-name}}</option>';
		$result = '<option value=""></option>';
		$all_groups = groups_get_groups(array('per_page' => null));
		foreach ($all_groups['groups'] as $group_object) {
			$code = str_replace('{{group-id}}', $group_object->id, $template);
			$code = str_replace('{{group-name}}', esc_html($group_object->name), $code);
			$result .= $code;
		}
		return $result;
	}
	public static function generate_shortcode_adder_code($params) {
		$option_code = '<option value="communityally-group" additional-input="communityally-group-id|communityally-group-tabs" ' .
				'accessally-shortcode-render-function="communityally_render_shortcode">CommunityAlly Discussion/Group Simple Display</option>' .
				'<option value="communityally-group-display" additional-input="communityally-group-id|communityally-group-tabs" ' .
				'accessally-shortcode-render-function="communityally_render_shortcode">CommunityAlly Discussion/Group Full Display</option>' .
				'<option value="communityally-members-display" additional-input="communityally-members-tabs" ' .
				'accessally-shortcode-render-function="communityally_render_shortcode">CommunityAlly Members Profile</option>';

		$additional_input_code = file_get_contents(dirname(__FILE__) . '/template/communityally-shortcode-additional-input-code.php');
		$communityally_group_selection = self::get_group_selection_code();
		$additional_input_code = str_replace('{{communityally-group-selection}}', $communityally_group_selection, $additional_input_code);

		$options = '<option disabled="disabled">----- Community -----</option>' . $option_code;
		$params['shortcode-option']['communityally'] = array(
			'priority' => 10,
			'code' => $options
		);
		$params['help-text'] .= file_get_contents(dirname(__FILE__) . '/template/communityally-shortcode-helptext.php');
		$params['additional-input'] .= $additional_input_code;
		return $params;
	}
	// </editor-fold>

	// <editor-fold defaultstate="collapsed" desc="Enqueue Script for AA Gutenberg block">
	public static function enqueue_scripts_for_gutenberg_blocks( $dependency_script_list, $accessally_styling_settings ) {
		$min = bp_core_get_minified_asset_suffix();
		wp_enqueue_style('bp-nouveau', buddypress()->plugin_url . "bp-templates/bp-nouveau/css/buddypress{$min}.css", array(), bp_get_version() );
		wp_add_inline_style('bp-nouveau', get_communityally_color_css_block($accessally_styling_settings));
		$dependency_script_list []= 'bp-nouveau';
		return $dependency_script_list;
	}
	public static function communityally_get_groups( $retval, $args ) {
		if (!isset($GLOBALS['bp'])) {
			$GLOBALS['bp'] = buddypress();
		}
		$bp = $GLOBALS['bp'];

		if (!isset($bp->groups)) {
			bp_setup_groups();
		}
		if (!isset($bp->table_prefix)) {
			$bp->table_prefix = bp_core_get_table_prefix();
		}
		if (!isset($bp->groups->table_name)) {
			$bp->groups->setup_globals();
		}

		$all_groups = groups_get_groups( $args );
		return $all_groups;
	}
	// </editor-fold>

	// <editor-fold defaultstate="collapsed" desc="Add shortcode options for AccessAlly Gutenberg block">
	public static function add_gutenberg_shortcode($config) {
		$config['communityally-group-display'] = array(
					'label' => 'CommunityAlly Discussion / Group Full Display',
					'code' => '[communityally_group_display group_id="%s" tab="%s"]',
					'input' => array(14, 'communityally-group-tab'),
			);
		$config['communityally-group'] = array(
					'label' => 'CommunityAlly Discussion / Group Simple Display',
					'code' => '[communityally_group group_id="%s" tab="%s"]',
					'input' => array(14, 'communityally-group-tab'),
			);
		$config['communityally-members'] = array(
					'label' => 'CommunityAlly Member Profile',
					'code' => '[communityally_members_display tab="%s"]',
					'input' => array('communityally-members-tab'),
			);
		return $config;
	}
	public static function add_gutenberg_additional_selection_options($all_selections) {
		$group_selection = self::communityally_get_groups(false, array());
		$all_group_selection = array(
			array('label' => 'Current page', 'value' => '0'),
		);
		if (!empty($group_selection)){
			foreach ($group_selection['groups'] as $group){
				$all_group_selection []= array('label' => $group->name, 'value' => $group->id);
			}
		}
		$all_selections[14] = array('label' => 'communityally group', 'selection' => $all_group_selection);

		$group_tabs = array(
			array('label' => 'Use on the Group page', 'value' => ''),
			array('label' => 'Feed', 'value' => 'activity'),
			array('label' => 'Discussions', 'value' => 'discussion'),
			array('label' => 'Member list', 'value' => 'members'),
			array('label' => 'Settings', 'value' => 'admin'),
			);
		$all_selections['communityally-group-tab'] = array('label' => 'Tab', 'selection' => $group_tabs);

		$member_profile_tabs = array(
			array('label' => 'Use on the Member Profile page', 'value' => ''),
			array('label' => 'Profile', 'value' => 'profile'),
			array('label' => 'Favorites', 'value' => 'activity'),
			array('label' => 'Messages - Inbox', 'value' => 'messages-inbox'),
			array('label' => 'Messages - Starred', 'value' => 'messages-starred'),
			array('label' => 'Messages - Sentbox', 'value' => 'messages-sent'),
			array('label' => 'Notifications - Unread', 'value' => 'notifications-unread'),
			array('label' => 'Notifications - Read', 'value' => 'notifications-read'),
			array('label' => 'Group list', 'value' => 'groups'),
			array('label' => 'Settings', 'value' => 'settings'),
			);
		$all_selections['communityally-members-tab'] = array('label' => 'Tab', 'selection' => $member_profile_tabs);

		return $all_selections;
	}
	// </editor-fold>

	// <editor-fold defaultstate="collapsed" desc="Redirect to the edit page for special pages">
	public static function ajax_special_pages_edit_redirect() {
		if (!isset($_REQUEST['type']) || !isset($_REQUEST['page_id'])) {
			echo 'Invalid edit URL';
			die();
		}
		if ($_REQUEST['page_id'] > 0) {
			$redirect_url = get_edit_post_link($_REQUEST['page_id'], '');

			wp_redirect(esc_url_raw($redirect_url));
			exit;
		}
		echo 'Invalid edit URL';
		die();
	}
	// </editor-fold>

	// <editor-fold defaultstate="collapsed" desc="Utility: return if the private messaging feature is enabled">
	public static function is_messaging_enabled() {
		$active_components = bp_get_option( '_communityally_active_components' );
		if (!isset($active_components['messages'])){
			return false;
		}
		return true; // check if Private messages settings is enabled
	}
	// </editor-fold>
}
add_filter('communityally_get_groups', array('CommunityAllySettings', 'communityally_get_groups'), 10, 2);
add_filter('accessally_enqueue_scripts_for_gutenberg_blocks', array('CommunityAllySettings', 'enqueue_scripts_for_gutenberg_blocks'), 10, 2);
add_filter('accessally_block_get_shortcode_list', array('CommunityAllySettings', 'add_gutenberg_shortcode'), 10, 1);
add_filter('accessally_block_generate_additional_selection_options', array('CommunityAllySettings', 'add_gutenberg_additional_selection_options'), 10, 1);
add_action('accessally_enqueue_shortcode_render_script', array('CommunityAllySettings', 'enqueue_shortcode_render_script'), 10, 1);
add_filter('accessally_shortcode_adder_options', array('CommunityAllySettings', 'generate_shortcode_adder_code'), 10, 1);

add_filter('accessally_add_submenu', array('CommunityAllySettings', 'add_accessally_submenu_pages'));

add_action('admin_enqueue_scripts', array('CommunityAllySettings', 'enqueue_administrative_resources'));
add_action('wp_ajax_communityally_settings_save_page', array('CommunityAllySettings', 'ajax_save_page_settings'));

// <editor-fold defaultstate="collapsed" desc="add action hooks">
add_action('bp_init', array('CommunityAllySettings', 'add_actions'), 10);
// </editor-fold>